/*********************************************************************
 *
 *      Microchip USB C18 Firmware -  USB Bootloader Version 1.00
 *
 *********************************************************************
 * FileName:        boot.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the Company) for its PICmicro Microcontroller is intended and
 * supplied to you, the Companys customer, for use solely and
 * exclusively on Microchip PICmicro Microcontroller products. The
 * software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Rawin Rojvanit       11/19/04    Original. USB Bootloader
 ********************************************************************/

/******************************************************************************
 * -boot.c-
 * This file contains functions necessary to carry out bootloading tasks.
 * The only 2 USB specific functions are BootInitEP() and BootService().
 * All other functions can be reused with other communication methods.
 *****************************************************************************/

/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include "typedefs.h"
#include "usb.h"
#include "io_cfg.h"
#include "carscroll.h"
//
extern char updatingVariable;
extern ram char tempbuffer[TEMPBUFF_SIZE];
extern byte screenOffPeriod;
extern byte screenOnPeriod;
extern int cursor;
extern OUTPUT output[NUM_OUTPUTS];
//
extern byte reqReset;
extern byte pushVariable;
int WriteEEPROM(int, int);
int ReadEEPROM(int);
void putString(int, int);
void getString(int, int);
void putVariableMode(int, int);
void putVariableUpdatePeriod(int, int);
void setCurrentBrightness(int);
void setCurrentScrollSpeed(int);
void setScrollSpeed(int);
void setBrightness(int);
//
extern VARIABLE variable[NUM_VARS];
extern int updateVariable(int);
void blankDisplay(void);
extern int screenCounter;
extern MAPPOINT freezePoint;
/** V A R I A B L E S ********************************************************/
byte counter;
byte byteTemp;
byte trf_state;
word big_counter;
/** P R I V A T E  P R O T O T Y P E S ***************************************/

/** D E C L A R A T I O N S **************************************************/
#pragma code

/** C L A S S  S P E C I F I C  R E Q ****************************************/

/** U S E R  A P I ***********************************************************/

/******************************************************************************
 * Function:        void BootInitEP(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        BootInitEP initializes bootloader endpoints, buffer
 *                  descriptors, internal state-machine, and variables.
 *                  It should be called after the USB host has sent out a
 *                  SET_CONFIGURATION request.
 *                  See USBStdSetCfgHandler() in usb9.c for examples.
 *
 * Note:            None
 *****************************************************************************/
void BootInitEP(void)
{   
    trf_state = WAIT_FOR_CMD;
    BOOT_UEP = EP_OUT_IN|HSHK_EN;               // Enable 2 data pipes

    /*
     * Do not have to init Cnt of IN pipes here.
     * Reason:  Number of bytes to send to the host
     *          varies from one transaction to
     *          another. Cnt should equal the exact
     *          number of bytes to transmit for
     *          a given IN transaction.
     *          This number of bytes will only
     *          be known right before the data is
     *          sent.
     */
    BOOT_BD_OUT.Cnt = sizeof(dataPacket);   // Set buffer size
    BOOT_BD_OUT.ADR = (byte*)&dataPacket;   // Set buffer address
    BOOT_BD_OUT.Stat._byte = _USIE|_DAT0|_DTSEN;// Set status

    BOOT_BD_IN.ADR = (byte*)&dataPacket;    // Set buffer address
    BOOT_BD_IN.Stat._byte = _UCPU|_DAT1;    // Set buffer status

}//end BootInitEP

void StartWrite(void)
{
    /*
     * A write command can be prematurely terminated by MCLR or WDT reset
     */
	GBIT=1;						// blank the display!
	INTCONbits.GIE=0;			// disable interrupt!
	
    EECON2 = 0x55;
    EECON2 = 0xAA;
    EECON1_WR = 1;
	INTCONbits.GIE=INTERRUPT_ENABLE; 		// reenable Interrupts
}//end StartWrite

void ReadProgMem(MEM_PTR ptr, int doffset, int length) //TESTED: Passed
{
	//
	// reads into tempbuffer[doffset] up to length bytes from program memory at ptr
	//
    for (counter = 0; counter < length; counter++)
    {
        byteTemp = *(ptr.pAdr++);
        if((doffset>=0)&&(doffset<TEMPBUFF_SIZE))tempbuffer[doffset++]=byteTemp;
    }//end for
}//end ReadProgMem

void WriteProgMem(MEM_PTR ptr, int doffset, int length) //TESTED: Passed
{
	// 
	// writes from tempubffer[doffset] up to length bytes to program memory at ptr
	//
	// ptr must be on 32 byte boundary...
    EECON1 = 0b10000100;     		//Setup writes: EEPGD=1,WREN=1
    //LEN = # of byte to write
    for (counter = 0; counter < length; counter++)
    {
        if((doffset>=0)&&(doffset<TEMPBUFF_SIZE))*ptr.pAdr++= tempbuffer[doffset++];
        if ((counter & 0b00011111) == 0b00011111)
        {
            StartWrite();
        }
		//end if
   }	//end for
}//end WriteProgMem

void EraseProgMem(MEM_PTR ptr) //TESTED: Passed
{
	// erases one block of 64 bytes at ptr
	// ptr must be on 64 byte boundary
    EECON1 = 0b10010100;    //Setup writes: EEPGD=1,FREE=1,WREN=1
	*ptr.pAdr;
    StartWrite();
}//end EraseProgMem

int WriteEEPROM(int address, int data)
{
		int i;
		i=0xFF & ReadEEPROM(address);
		if(i!=(data & 0xFF))
		{
		EEADR=(byte)address;
//        EEADRH =(byte)(address>>8);
        EEDATA = data;
        EECON1 = 0b00000100;    //Setup writes: EEPGD=0,WREN=1

        StartWrite();
        while(EECON1_WR!=0);       //Wait till WR bit is clear
		}
		return ReadEEPROM(address);
}

int ReadEEPROM(int address)
{
    EECON1 = 0x00;
    EEADR =  (byte)address;
//	EEADRH = (byte)(address>>8);
    EECON1_RD = 1;
    return EEDATA;
}

int myWriteProgMemUpTo32(MEM_PTR ptr, int doffset, int length)
{
	//
	// this subroutine tries to write the following to program memory
	// dataPacket.data[doffset]				= start of contiguous data
	// ptr               				    = contains the destination address in program memory
	// length  				                = number of bytes to write (<=32 bytes)
	//
	// the other memory around is preserved etc.
	//
	// uses tempbuffer as temporary RAM
	//
	MEM_PTR iptr;
	int i, offset;

	if(length<0)length=0; else if(length>31)length=32;
	iptr.low=ptr.low;
	iptr.high=ptr.high;
	iptr.upper=ptr.upper;
	ptr.low&=0xC0;							// force 64 byte boundary;
	offset=(iptr.low-ptr.low);			 	// the offset produced by forcing boundary
	ReadProgMem(ptr, 0, 64);				// now tempbuffer contains the two adjacent blocks
	for(i=0; i<length; i++)
	{
	tempbuffer[offset+i]=dataPacket.data[doffset+i];
	}							// copy the data to the temporary store
	EraseProgMem(ptr);			// erase 64 byte block
	WriteProgMem(ptr,0,64);		// write the two blocks
	return length;				// returns the number of bytes written...
}

int myReadProgMemUpTo32(MEM_PTR ptr, int doffset, int length)
{
	//
	// this subroutine tries to read the following to program memory
	// dataPacket.data[doffset]		= start of data to write to
	// ptr          			    = contains the source address in program memory
	// length		                = number of bytes to read (<=32 bytes)
	//
	// uses tempbuffer as temporary RAM
	//
	int i, offset;
	MEM_PTR iptr;
	
	if(length<0)length=0; else if(length>31)length=32;
	iptr.low=ptr.low;
	iptr.high=ptr.high;
	iptr.upper=ptr.upper;
	ptr.low&=0xC0;							// force 64 byte boundary;
	offset=+(iptr.low-ptr.low);	 	// the offset produced by forcing boundary
	ReadProgMem(ptr, 0, 64);				// read up to 64 bytes
	for(i=0; i<length; i++)
	{
	dataPacket.data[doffset+i]=tempbuffer[offset+i];
	}
	return length;					// return the number of bytes read...
}


/*
void prepareInfoPacket(void)
{
	counter=BOOT_EP_SIZE;
}
*/


int doRWRam(ram far byte* sou, int len, byte dir)
{
	// if dir=READ_DIR copies from source to destination
	// if dir=WRITE_DIR copies from destination to source
	int i;
	ram far byte* dest;

	dest=&dataPacket.data[0];
	i=0;
	while(i<len)
	{
	if(dir==READ_DIR)*dest++=*sou++; else *sou++=*dest++;
	i++;
	}
	counter=BOOT_EP_SIZE;
	return len;
}

int doRWRom(MEM_PTR sou, int len, byte dir)
{
	int i, j;
	//
	i=len;
	j=0;
	while(i>=32)
	{
	if(dir==READ_DIR)myReadProgMemUpTo32(sou,j,32); else myWriteProgMemUpTo32(sou, j, 32);
	i-=32;
	j+=32;
	sou.pAdr+=32;
	}
	if(i>0){ if(dir==READ_DIR)myReadProgMemUpTo32(sou,j,i); else myWriteProgMemUpTo32(sou, j, i); }
	j+=i;
	counter=BOOT_EP_SIZE;
	return j;
}

int doRWEE(int sou, ram byte *dest, int len, byte dir)
{
	int i, j;
	//
	i=len;
	j=0;
	while(i>0)
	{
	if(dir==READ_DIR)*dest=ReadEEPROM(sou); else WriteEEPROM(sou,*dest);
	i--;
	dest++;
	sou++;
	}
	counter=BOOT_EP_SIZE;
	return j;
}

void BootService(void)
{
	unsigned int i, j, k;


    if(usb_device_state < CONFIGURED_STATE) return;
    
    if(trf_state == SENDING_RESP)
    {
        if(!mBootTxIsBusy())
        {
            BOOT_BD_OUT.Cnt = sizeof(dataPacket);
            mUSBBufferReady(BOOT_BD_OUT);
            trf_state = WAIT_FOR_CMD;
        }//end if
        return;
    }
	//end if
    
	GBIT=1;
    if(!mBootRxIsBusy())
    {
        counter = 0;
        switch(dataPacket.CMD)
        {
			case CAR_DISPLAY_RW_RAM:
				doRWRam(dataPacket.ADRA.pAdrRam, dataPacket.len, dataPacket.dir);
				break;

			case CAR_DISPLAY_RW_ROM:
				doRWRom(dataPacket.pAdr, dataPacket.len, dataPacket.dir);
				break;

			case CAR_DISPLAY_RW_EE:
				doRWEE(dataPacket.iarg, &dataPacket.data[0], dataPacket.len, dataPacket.dir);
				break;

			case CAR_DISPLAY_RESET:
				reqReset=1;
				counter=BOOT_EP_SIZE;
				break;

/*
			case CAR_DISPLAY_SET_MODE:
				putVariableMode(dataPacket.varn, dataPacket.mode);
				prepareInfoPacket();
				break;

			case CAR_DISPLAY_SET_SCROLL_SPEED:
				setScrollSpeed(dataPacket.arg);
				prepareInfoPacket();
				break;

			case CAR_DISPLAY_SET_BRIGHTNESS:
				setBrightness(dataPacket.arg);
				prepareInfoPacket();
				break;
			
			case CAR_DISPLAT_SET_DISPLAY_OBJECT:
				setDisplayObject(dataPacket.arg);
				prepareInfoPacket();
				break;

*/

			default:
				counter=0;
                break;
        }//end switch()
        trf_state = SENDING_RESP;
        if(counter != 0)
        {
            BOOT_BD_IN.Cnt = counter;
            mUSBBufferReady(BOOT_BD_IN);
        }//end if
    }//end if
}//end BootService

/******************************************************************************
 * Function:        void BlinkUSBStatus(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        BlinkUSBStatus turns on and off LEDs corresponding to
 *                  the USB device state.
 *
 * Note:            mLED macros can be found in io_cfg.h
 *                  usb_device_state is declared in usbmmap.c and is modified
 *                  in usbdrv.c, usbctrltrf.c, and usb9.c
 *****************************************************************************/
/*
void BlinkUSBStatus(void)
{
    static word led_count=0;
    
    if(led_count == 0)led_count = 20000U;
    led_count--;

    #define mLED_Both_Off()         {mLED_1_Off();mLED_2_Off();}
    #define mLED_Both_On()          {mLED_1_On();mLED_2_On();}
    #define mLED_Only_1_On()        {mLED_1_On();mLED_2_Off();}
    #define mLED_Only_2_On()        {mLED_1_Off();mLED_2_On();}

    if(UCONbits.SUSPND == 1)
    {
        if(led_count==0)
        {
  //          mLED_1_Toggle();
//            mLED_2 = mLED_1;        // Both blink at the same time
        }//end if
    }
    else
    {
        if(usb_device_state == DETACHED_STATE)
        {
    //        mLED_Both_Off();
        }
        else if(usb_device_state == ATTACHED_STATE)
        {
      //      mLED_Both_On();
        }
        else if(usb_device_state == POWERED_STATE)
        {
        //    mLED_Only_1_On();
        }
        else if(usb_device_state == DEFAULT_STATE)
        {
          //  mLED_Only_2_On();
        }
        else if(usb_device_state == ADDRESS_STATE)
        {
            if(led_count == 0)
            {
            //    mLED_1_Toggle();
              //  mLED_2_Off();
            }//end if
        }
        else if(usb_device_state == CONFIGURED_STATE)
        {
            if(led_count==0)
            {
               // mLED_1_Toggle();
              //  mLED_2 = !mLED_1;       // Alternate blink                
            }//end if
        }//end if(...)
    }//end if(UCONbits.SUSPND...)

}//end BlinkUSBStatus
*/
/** EOF boot.c ***************************************************************/
